from flask import flash, redirect, request, url_for
from mail_utils import send_verification_email
from utils import create_initial_users, recreate_user_table, repopulate_users_from_csv, write_users_to_csv
from models import User, db
from werkzeug.security import generate_password_hash
import logging
# --- START OF CHANGES ---
from datetime import datetime, timezone # Import timezone
# --- END OF CHANGES ---

def create_user(session, username, email, password):
    """
    Creates a new user, adds to session, and handles potential errors.
    Returns the new User object on success, or None on failure.
    Failure reasons: duplicate username/email, password too short, database error.
    """
    # Basic validation (can be enhanced)
    if not username or not email or not password:
        logging.warning("User creation failed: Missing required fields.")
        return None # Indicate failure: Missing fields

    if len(password) < 6:
        logging.warning(f"User creation failed for {username}: Password too short.")
        return None # Indicate failure: Password too short

    # Check for existing user
    existing_user = User.query.filter(
        (User.username == username) | (User.email == email)
    ).first()
    if existing_user:
        logging.warning(f"User creation failed: Username '{username}' or email '{email}' already exists.")
        return None # Indicate failure: Duplicate user

    # Hash password and create user
    hashed_password = generate_password_hash(password)
    try:
        new_user = User(username=username, email=email, password=hashed_password, confirmed=False)
        session.add(new_user)
        session.commit() # Commit here to get the user ID if needed immediately

        # Actions after successful commit
        write_users_to_csv(session) # Consider if this needs the session or just db
        send_verification_email(username, email) # Send verification email

        logging.info(f"User '{username}' created successfully.")
        return new_user # Return the created user object

    except Exception as e:
        logging.error(f"Error creating user '{username}': {e}", exc_info=True)
        session.rollback()
        return None # Indicate failure: Database or other error

# ----------------------------------------------------------
# REMOVE the create_user_route() function definition from here
# ----------------------------------------------------------

# ... (keep other functions like edit_user, save_users, load_users, etc. for now)

# Add this function to your users.py file
def delete_unverified_inactive_users():
    # ... (keep existing implementation) ...
    pass # Placeholder if you copied the whole file, keep the original code here


# --- START OF CHANGES in users.py ---
def get_user_by_id(user_id):
    """Fetches a user by their ID using db.session.get."""
    # Use the modern db.session.get() method which is often preferred for PK lookups
    return db.session.get(User, user_id)

# Modify update_user_details signature and logic
# Remove last_update_from_form parameter
def update_user_details(user_id, username, email, confirmed, login_count, last_login_from_form):
    """
    Updates user details in the database. Automatically sets last_update.
    Returns True on success, False on failure.
    """
    # Use db.session.get for fetching by primary key
    user = db.session.get(User, user_id) # Changed from get_user_by_id for consistency within function
    if not user:
        logging.warning(f"Attempted to update non-existent user with ID {user_id}.")
        return False # User not found

    try:
        # Check for username duplicates (excluding the current user)
        if username != user.username:
            existing_user_username = User.query.filter(User.username == username, User.id != user_id).first()
            if existing_user_username:
                logging.warning(f"Update failed for user ID {user_id}: Username '{username}' already exists.")
                # Optionally, flash a message here or return a specific error code
                # flash(f"Username '{username}' is already taken.", "danger")
                return False # Username conflict

        # Check for email duplicates (excluding the current user)
        if email != user.email:
            existing_user_email = User.query.filter(User.email == email, User.id != user_id).first()
            if existing_user_email:
                logging.warning(f"Update failed for user ID {user_id}: Email '{email}' already exists.")
                # Optionally, flash a message here or return a specific error code
                # flash(f"Email '{email}' is already registered.", "danger")
                return False # Email conflict

        # Update user attributes
        user.username = username
        user.email = email
        user.confirmed = confirmed
        # Ensure login_count is an integer (it should be from the route)
        user.login_count = int(login_count) if login_count is not None else 0
        # last_login_from_form can be None if Optional() was used and field was empty
        user.last_login = last_login_from_form
        # Automatically update the last_update timestamp
        user.last_update = datetime.now(timezone.utc) # Use UTC timezone

        db.session.commit()
        logging.info(f"User details updated successfully for user ID {user_id}.")
        # Optionally update CSV after successful commit
        # write_users_to_csv(db.session) # Pass the session if needed
        return True # Indicate success

    except Exception as e:
        db.session.rollback()
        logging.error(f"Error updating user details for user ID {user_id}: {e}", exc_info=True)
        return False # Indicate failure
# --- END OF CHANGES in users.py ---



def save_users():
    write_users_to_csv(db.session)
    return redirect(url_for("admin"))


def load_users():
    if request.method == "POST":
        repopulate_users_from_csv(db)
        flash("Users loaded successfully!", "success")
        return redirect(url_for("admin"))


def recreate_user():
    if request.method == "POST":
        write_users_to_csv(db.session)
        recreate_user_table(db)
        create_initial_users(db)
        flash("User table recreated successfully!", "success")
        return redirect(url_for("admin"))


# Add this function to your users.py file
def delete_unverified_inactive_users():
    """
    Deletes users from the database who meet the following criteria:
    1. Email is not confirmed (user.confirmed is False).
    2. Login count is zero (user.login_count is 0).

    Flashes a message indicating the outcome (success, no users found, or error).
    Returns the number of users deleted, or 0 if none were found or an error occurred.
    """
    num_deleted = 0
    try:
        # Query for users matching both criteria
        users_to_delete = User.query.filter(
            User.confirmed == False,
            User.login_count == 0
        ).all() # Use .all() to get a list of users to delete

        num_deleted = len(users_to_delete)

        if not users_to_delete:
            flash("No unverified users with zero logins found to delete.", "info")
            return 0

        # Iterate and delete each user found
        for user in users_to_delete:
            db.session.delete(user)

        # Commit the changes to the database
        db.session.commit()
        flash(f"Successfully deleted {num_deleted} unverified, inactive user(s).", "success")
        # Optionally, write to CSV again if you want the backup updated immediately
        # write_users_to_csv(db)

    except Exception as e:
        db.session.rollback() # Important: Rollback transaction on error
        logging.error(f"Error deleting unverified inactive users: {e}", exc_info=True) # Log the full error
        flash("An error occurred while deleting inactive users. Please check the server logs.", "danger")
        num_deleted = 0 # Ensure 0 is returned on error

    return num_deleted

