from openai import OpenAI
import os
import logging


def setup_logger(logs_dir="logs"):
    """Sets up the logger with proper error handling."""
    try:
        log_path = os.path.join(os.path.dirname(__file__), logs_dir) # Correct path
        os.makedirs(log_path, exist_ok=True)

        log_file_path = os.path.join(log_path, 'ai_processing.log')
        logger = logging.getLogger('ai_processor')  # Standard logger name
        logger.setLevel(logging.INFO)  # Set level for the logger

        if not logger.handlers:  # Check if handler already exists
            handler = logging.FileHandler(log_file_path, encoding='utf-8')
            handler.setLevel(logging.INFO) 
            formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(filename)s - %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)

    except Exception as e:  # Catch broader exceptions during setup
        print(f"Error setting up logging: {e}", file=sys.stderr) # Important for debugging
        return None # Indicate setup failure

    return logger


# Call the setup_logging function at the top level 
ai_logger = setup_logger() # Pass in log filename


def call_AI(prompt, model=None, temperature=None, max_tokens=None):
    """
    Calls the OpenAI API with the given prompt and parameters.

    Args:
        prompt (str): The prompt to send to the OpenAI API.
        model (str, optional): The OpenAI model to use. Defaults to None (uses environment variable or falls back to a default).
        temperature (float, optional): The temperature for the API call. Defaults to None (uses environment variable or falls back to a default).
        max_tokens (int, optional): The maximum number of tokens for the response. Defaults to None (uses environment variable or falls back to a default).

    Returns:
        str: The generated response from the OpenAI API.
        None: If there is an error during the API call.

    Raises:
        ValueError: If the OpenAI API key is not set.
    """
    try:
        # Get the OpenAI API key from environment variables
        openai_api_key = os.environ.get('OPENAI_API_KEY')

        # Check if the API key is set
        if not openai_api_key:
            raise ValueError("OPENAI_API_KEY environment variable not set.")

        # Initialize the OpenAI client
        client = OpenAI(api_key=openai_api_key)

        # Set default values if parameters are not provided
        if model is None:
            model = os.environ.get('OPENAI_MODEL', 'gpt-4o')  # Default to gpt-4 if not specified

        if temperature is None:
            temperature = float(os.environ.get('OPENAI_TEMPERATURE', 0.5))  # Default to 0.7

        if max_tokens is None:
            max_tokens = int(os.environ.get('OPENAI_MAX_TOKENS', 15000))

        # Validate temperature value
        if not 0 <= temperature <= 1:
            ai_logger.warning(f"Temperature value {temperature} is outside the allowed range [0, 2]. Using default value 0.7.")
            temperature = 0.7 # Use configured value or default 0.7

        response = client.chat.completions.create(
            model=model,
            messages=[
                {"role": "system", "content": "You are a brilliant financial analyst at a major wall street firm."},
                {"role": "user", "content": prompt},
            ],
            max_tokens=max_tokens,
            temperature=temperature,
        )

        # Extract and return the generated answer
        answer = response.choices[0].message.content.strip()
        return answer

    except OpenAI.APIError as e:
        ai_logger.error(f"OpenAI API error: {e}")
        return None  # Or handle the error as needed
    except ValueError as e:  # Catch missing API Key Value Error
        ai_logger.error(f"Error: {e}")
        return None
    except Exception as e:
        ai_logger.error(f"An unexpected error occurred: {e}")
        return None