from flask import render_template
from flask import (
    render_template,
    request,
    redirect,
    url_for,
    flash,
    redirect,
    session,
)
from flask_login import (
    login_user,
    login_required,
    logout_user,
)
from forms import (
    RegistrationForm,
    LoginForm,
    ResetPasswordForm,
    UpdatePasswordForm,
    OptionsForm,
    UpdateUserForm,
    ReverifyEmailForm,
)
from utils import write_users_to_csv, get_call_count, get_top_calls, get_unique_symbols
from mail_utils import  send_email, send_verification_email, resend_verification_email
from models import db, User, Call
from itsdangerous import SignatureExpired
from werkzeug.security import generate_password_hash, check_password_hash
from forms import RegistrationForm, LoginForm
from datetime import datetime, timezone
import users

# --- START OF CHANGES in index view ---
# Import the new utility function
from utils import write_users_to_csv, get_index_page_data # <-- Added get_index_page_data

# ... (other imports remain the same)

def index():
    #recreate_user_table(db)
    #create_initial_users(db)
    # Original logic:
    # num_calls = get_call_count(db)
    # num_symbols = get_unique_symbols(db)
    # top_calls = get_top_calls(db, 5)
    # return render_template("index.html", calls=top_calls, num_calls=num_calls, num_symbols=num_symbols)

    # New logic: Call the utility function
    index_data = get_index_page_data(db, 5) # Pass db and desired number of top calls

    return render_template(
        "index.html",
        calls=index_data["top_calls"],
        num_calls=index_data["num_calls"],
        num_symbols=index_data["num_symbols"]
    )
# --- END OF CHANGES in index view ---



def login():
    form = LoginForm()
    if form.validate_on_submit():
        user = User.query.filter_by(username=form.username.data).first()
        if user and check_password_hash(user.password, form.password.data):
            if user.confirmed:
                # Update last_login before logging in
                user.last_ip = request.remote_addr
                user.last_login = datetime.now(timezone.utc)
                user.login_count += 1
                db.session.commit()  # Commit the change
                login_user(user, remember=form.remember)
                return redirect(url_for("dashboard"))
            else:
                flash("Please confirm your email address before logging in.", "warning")
                return redirect(url_for("resend_verification"))
        else:
            flash("Invalid username or password.", "danger")
    return render_template("login.html", title="Login", form=form)


@login_required
def logout():
    logout_user()
    return redirect(url_for("index"))


from werkzeug.security import generate_password_hash

def register():
    form = RegistrationForm()
    if form.validate_on_submit():
        username = form.username.data
        email = form.email.data
        password = form.password.data

        existing_user = User.query.filter(
            (User.username == username) | (User.email == email)
        ).first()
        if existing_user:
            flash(
                "A user with that username or email already exists. Please choose another.",
                "warning",
            )
            return redirect(url_for("register"))

        hashed_password = generate_password_hash(password) # Use default - assumed to be scrypt.
        new_user = User(username=username, email=email, password=hashed_password)
        db.session.add(new_user)
        db.session.commit()
        write_users_to_csv(db) # Update the CSV with the new user data.
        send_verification_email(username, email)  # Send verification email *after* commit
        flash(
            "A confirmation email has been sent to your email address. Please check your inbox.",
            "info",
        )
        return redirect(url_for("login"))  # Redirect after successful registration
    return render_template("register.html", title="Register", form=form)


@login_required
def delete_user(user_id):
    user = User.query.get(user_id)
    if user:
        db.session.delete(user)
        db.session.commit()
        flash(f"User {user_id} has been deleted.", "success")
    else:
        flash(f"User {user_id} not found.", "danger")
    return redirect(url_for("show_users"))


def delete_selected_users():
    user_ids = request.form.getlist('user_ids') # Gets list of selected user IDs.
    if user_ids:
        for user_id in user_ids:
            try:
                user = User.query.get(int(user_id))  # Assuming User model exists
                if user:
                    db.session.delete(user)
            except ValueError: # Handle cases where user_id is not an integer
                pass # Or flash an error message
        db.session.commit()
    return redirect(url_for('show_users'))


def send_password_reset():
    from app import s
    form = ResetPasswordForm()
    if form.validate_on_submit():
        email = form.email.data
        user = User.query.filter_by(email=email).first()
        if user:
            subject = "Password Reset Email"
            token = s.dumps(email, salt="password-reset")
            body = f'Please click the link to reset your password: {url_for("update_password", token=token, _external=True)}'
            send_email(email, subject, body)
            flash(
                "A password reset email has been sent to your email address. Please check your inbox.",
                "info",
            )
            return redirect(url_for("login"))
        else:
            flash("No user found with that email address.", "danger")
    return render_template("reset_password.html", title="Login", form=form)


def process_password_reset(token):
    from app import s
    try:
        email = s.loads(token, salt="password-reset", max_age=86400)  # Load email from token
    except SignatureExpired:
        flash("The password reset link has expired. Please request a new one.", "danger")
        return redirect(url_for("send_password_reset"))  # Redirect to password reset request page

    form = UpdatePasswordForm()
    if form.validate_on_submit():
        user = User.query.filter_by(email=email).first()
        if user:
            user.password = generate_password_hash(form.password.data) # Use default - assumed to be scrypt
            db.session.commit()
            flash("Your password has been updated. You can now log in.", "success")
            return redirect(url_for("login"))
        else:
            flash("No user found with that email address.", "danger") # Handle user not found
            return redirect(url_for("login"))  # Redirect to login
    return render_template("update_password.html", form=form, token=token)  # Pass the token back to the form


def resend_verification():
    form = ReverifyEmailForm()
    if form.validate_on_submit():
        resend_verification_email(form.email.data)
    else:
        print(form.errors)
    return render_template("reverify.html", title="Reverify Email", form=form)


def options():
    form = OptionsForm()
    if request.method == "GET":
        print("Model=", session["model"])
        form.model.data = session["model"]  # 1 = 'gpt-3.5-turbo', 2 = 'gpt-4'
        form.temperature.data = session["temperature"]
    if form.validate_on_submit():
        session["model"] = form.model.data
        session["temperature"] = float(form.temperature.data)
        flash("Options Saved!", "info")
        return redirect(url_for("dashboard"))
    return render_template("/GPT/options.html", form=form)


# --- START OF CHANGES in edit_user_route ---
@login_required # Add decorator if only logged-in users (admins?) can edit
# @admin_required # Optional: Add if you have an admin check decorator
def edit_user_route(user_id):
    """Handles the web route for editing an existing user."""
    user = users.get_user_by_id(user_id) # Use the helper function
    if not user:
        flash(f"User with ID {user_id} not found.", "danger")
        return redirect(url_for("show_users")) # Assuming 'show_users' is the route for the user list

    # Initialize form. Pass obj=user for GET requests to pre-populate.
    form = UpdateUserForm(obj=user)

    # Explicitly set the default value for email_confirmed based on user data for GET requests
    if request.method == 'GET':
        form.email_confirmed.data = '1' if user.confirmed else '0'
        # Pre-populate last_login if it exists, WTForms handles the format conversion
        # obj=user should handle this, but explicit setting can be a fallback if needed
        # if user.last_login:
        #    form.last_login.data = user.last_login

    if form.validate_on_submit():
        # Process the form data on POST
        username = form.username.data
        email = form.email.data
        # Convert '1'/'0' string to boolean
        confirmed = bool(int(form.email_confirmed.data))
        # WTForms handles IntegerField conversion, default to 0 if None/empty after Optional()
        login_count = form.login_count.data if form.login_count.data is not None else 0
        # WTForms handles DateTimeField conversion, will be None if empty and Optional() is used
        last_login_value = form.last_login.data

        # Call the core logic function (ensure update_user_details signature matches)
        success = users.update_user_details(
            user_id=user_id,
            username=username,
            email=email,
            confirmed=confirmed,
            login_count=login_count,
            last_login_from_form=last_login_value, # Pass the datetime object or None
            # last_update is handled automatically in update_user_details now
        )

        if success:
            flash(f"User '{username}' (ID: {user_id}) updated successfully.", "success")
            return redirect(url_for("show_users")) # Redirect after successful update
        else:
            # Error message comes from update_user_details logging, flash generic here
            flash(f"Error updating user '{username}'. Username or email might already exist, or a database error occurred. Check logs.", "danger")
            # Re-render the form with validation errors if any

    # Render the template for GET requests or if POST validation fails/update fails
    # Pass user again in case needed in the template beyond form fields
    return render_template("edit_user.html", form=form, user=user, title="Edit User")
# --- END OF CHANGES in edit_user_route ---


# +++ ADD THIS FUNCTION +++
@login_required # Keep login required if this is an admin action
def create_user_route():
    """Handles the web route for creating a new user (typically by an admin)."""
    form = RegistrationForm() # Assuming UserCreationForm from wtforms
    if form.validate_on_submit():
        # Call the core logic function from the users module
        new_user = users.create_user(db.session, form.username.data, form.email.data, form.password.data)

        if new_user:
            flash(f"Account for '{new_user.username}' created. A verification email has been sent.", 'success')
            # Redirect to admin dashboard or user list
            # Make sure 'admin' is the correct endpoint name
            return redirect(url_for('admin'))
        else:
            # Flash a generic error. The specific reason was logged in create_user.
            # You could enhance create_user to return error codes/messages
            # if you want more specific feedback here.
            flash('Error creating user. Check logs or ensure username/email is unique and password meets criteria.', 'danger')
            # No redirect, show the form again with errors if WTForms handles them,
            # or just show the generic flash message.

    # Render the template for GET requests or if form validation fails
    return render_template('create_user.html', form=form, title="Create User")
# +++ END OF ADDED FUNCTION +++



@login_required
# @admin_required # Optional: Uncomment or add if you have an admin check decorator
def cleanup_inactive_users_route():
    """
    Handles the POST request to delete unverified, inactive users
    and redirects back to the admin page.
    """
    users.delete_unverified_inactive_users() # Call the function that does the work
    return redirect(url_for('admin')) # Redirect to the admin page
