import csv
from dotenv import load_dotenv
import os
from models import Call, User
from werkzeug.security import generate_password_hash
from datetime import datetime


def recreate_all_tables(db):
    try:
        db.drop_all()
        db.create_all()
        db.session.commit()
    except Exception as e:
        print(f"An error occurred while recreating tables: {e}")
        raise  # Re-raise the exception to propagate it to the caller


def recreate_call_table(db):
    try:
        Call.__table__.drop(db.engine)
        Call.__table__.create(db.engine)
        db.session.commit()
    except Exception as e:
        print(f"An error occurred while recreating the Call table: {e}")
        raise  # Re-raise the exception to propagate it to the caller


def recreate_user_table(db):
    try:
        User.__table__.drop(db.engine)
        User.__table__.create(db.engine)
        db.session.commit()
    except Exception as e:
        print(f"An error occurred while recreating the User table: {e}")
        raise  # Re-raise the exception to propagate it to the caller


def get_call_table_size(db):
    return db.session.query(Call).count()


def get_user_table_size(db):
    return db.session.query(User).count()


def get_call_by_id(db, call_id):
    call = db.session.query(Call).filter_by(id=call_id).first()
    return call


def get_calls_by_company(db, symbol):
    # Fetch calls for the given company symbol, ordered by date descending
    return db.session.query(Call).filter_by(symbol=symbol).order_by(Call.date.desc()).all()


def get_unique_symbols(db):
    unique_symbols = db.session.query(Call.symbol).distinct().count()
    return unique_symbols


def get_call_count(db):
    total_calls = db.session.query(Call).count()
    return total_calls

# --- START OF CHANGES for index page data ---
def get_index_page_data(db, num_top_calls=5):
    """
    Fetches the data required for the index page.

    Args:
        db: The database session/object.
        num_top_calls (int): The number of top calls to retrieve.

    Returns:
        dict: A dictionary containing 'num_calls', 'num_symbols', and 'top_calls'.
    """
    num_calls = get_call_count(db)
    num_symbols = get_unique_symbols(db)
    top_calls = get_top_calls(db, num_top_calls)
    return {
        "num_calls": num_calls,
        "num_symbols": num_symbols,
        "top_calls": top_calls,
    }
# --- END OF CHANGES for index page data ---


def get_top_calls(db, num_calls=5):
    top_calls = db.session.query(Call).order_by(Call.date.desc()).limit(num_calls).all()
    return top_calls


def get_environmental_variables():
    try:
        dotenv_path = os.path.join(os.path.dirname(__file__), "sandia.env")
        load_dotenv(dotenv_path)
    except Exception as e:
        print(f"An error occurred: {e}")

    # Check if the operating system is Unix-like
    if os.name == "posix":
        print("Unix")
        env_vars = os.environ
    # Check if the operating system is Windows
    elif os.name == "nt":
        print("Windows")
        env_vars = os.environ.copy()
    # Return the environmental variables
    return env_vars


from werkzeug.security import generate_password_hash

def create_initial_users(db):
    hashed_password = generate_password_hash("Lindo123$")  # Use default - assumed to be scrypt
    users_data = [
        {"username": "jasmine", "email": "info@sandia.com", "confirmed": True},
        {"username": "maya", "email": "hola@sandia.com", "confirmed": True},
    ]
    for user_data in users_data:
        existing_user = User.query.filter_by(username=user_data['username']).first()
        if not existing_user:  # Check if user already exists
            new_user = User(password=hashed_password, **user_data)
            db.session.add(new_user)
    db.session.commit()


# --- CHANGE THIS FUNCTION DEFINITION AND USAGE ---
def write_users_to_csv(session, filename="users.csv"): # Changed 'db' to 'session'
    """Writes the contents of the User table to a CSV file using the provided session."""

    with open(filename, "w", newline="") as csvfile:
        writer = csv.writer(csvfile)
        # Write header row with all User model columns
        writer.writerow([
            "id", "username", "email", "password", "confirmed",
            "favorites", "created_date", "last_update", "login_count", "last_login", "last_ip" # Added last_ip if it's in your model
        ])
        # Use the passed session directly for the query
        for user in session.query(User).all(): # Changed 'db.session' to 'session'
            # Format date/time fields to strings for CSV writing
            created_date_str = user.created_date.strftime('%Y-%m-%d %H:%M:%S') if user.created_date else None
            last_update_str = user.last_update.strftime('%Y-%m-%d %H:%M:%S') if user.last_update else None
            last_login_str = user.last_login.strftime('%Y-%m-%d %H:%M:%S') if user.last_login else None

            writer.writerow([
                user.id, user.username, user.email, user.password, user.confirmed,
                user.favorites, created_date_str, last_update_str,
                user.login_count, last_login_str, user.last_ip # Added last_ip
            ])
# --- END OF CHANGE ---


def repopulate_users_from_csv(db, filename="users.csv"):
    """Reads user data from a CSV file and populates the User table,
    avoiding duplicates based on the 'username' field."""
    existing_usernames = set(u.username for u in db.session.query(User.username).all())

    with open(filename, "r") as csvfile:
        reader = csv.DictReader(csvfile)  # Use DictReader for header-based access
        for row in reader:
            username = row['username']
            if username not in existing_usernames:
            # Convert string dates using the same method in utils
                created_date = convert_date_to_iso(row['created_date']) if row['created_date'] else None # use convert_date_to_iso()
                last_update = convert_date_to_iso(row['last_update']) if row['last_update'] else None # use convert_date_to_iso()
                last_login = convert_date_to_iso(row['last_login']) if row['last_login'] else None # use convert_date_to_iso()


                new_user = User(
                    id=int(row['id']),
                    username=username,
                    email=row['email'],
                    password=row['password'],
                    confirmed=(row['confirmed'].lower() == 'true'),  # Convert to boolean
                    favorites=row['favorites'],
                    created_date=created_date,
                    last_update=last_update,
                    login_count=int(row['login_count']),
                    last_login=last_login
                )
                db.session.add(new_user)
                existing_usernames.add(username) 

    db.session.commit()


def convert_date_to_iso(date_str):
    """Converts a date string in the format "9 Nov 23" to "YYYY-MM-DD". """
    try:
        # Correct usage of strptime
        date_obj = datetime.strptime(date_str, "%d %b %y")  
        return date_obj.strftime("%Y-%m-%d")
    except ValueError:
        print(f"Invalid date format: {date_str}")
        return None