from datetime import datetime
import logging
import sys
from models import User, Call
from flask import flash
import os
from models import Call, User, Participant
from flask import flash
from bs4 import BeautifulSoup
import json
from utils import convert_date_to_iso


def setup_logger(logs_dir="logs"):
    """Sets up the transcript logger with proper error handling."""
    try:
        log_path = os.path.join(os.path.dirname(__file__), logs_dir) # Correct path
        os.makedirs(log_path, exist_ok=True)

        log_file_path = os.path.join(log_path, 'transcript_processing.log')
        logger = logging.getLogger('transcript_processor')  # Standard logger name
        logger.setLevel(logging.INFO)  # Set level for the logger

        if not logger.handlers:  # Check if handler already exists
            handler = logging.FileHandler(log_file_path, encoding='utf-8')
            handler.setLevel(logging.INFO) 
            formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(filename)s - %(message)s')
            handler.setFormatter(formatter)
            logger.addHandler(handler)

    except Exception as e:  # Catch broader exceptions during setup
        print(f"Error setting up logging: {e}", file=sys.stderr) # Important for debugging
        return None # Indicate setup failure

    return logger


# Call the setup_logging function at the top level 
transaction_logger = setup_logger() # Pass in log filename


def get_transcript_dir():
    """Determines the correct directory for transcripts based on the environment."""

    if os.name == 'posix':  # Check for Linux/Unix environment
        transcript_dir = os.path.join(os.path.dirname(__file__), 'updates')
    else:  # Assume Windows environment
        base_dir = os.environ.get('ONEDRIVE', 'C:\\Users\\marcc\\OneDrive') # Get OneDrive path from environment variable, default to a standard path
        transcript_dir = os.path.join(base_dir, 'Updates')
    return transcript_dir


def manage_transcripts(db, number_to_process, offset=0):
    transcript_dir = get_transcript_dir()
    transaction_logger.info(f"Transcript directory: {transcript_dir}")
    processed_symbols = []
    num_processed = 0
    num_skipped = 0
    bad_files = 0

    files = os.listdir(transcript_dir)[offset:]

    for filename in files:
        if filename.endswith(".txt"):
            num_processed += 1
            if num_processed > number_to_process:
                break

            if num_processed % 100 == 0:
                print(f"Processed {num_processed} files...")
                db.session.commit()

            file_path = os.path.join(transcript_dir, filename)

            try:
                result = parse_transcript(file_path)
            except Exception as e:
                # 5. Use the dedicated logger
                transaction_logger.error(f"Error parsing file: {filename}, Error: {str(e)}")
                bad_files += 1
                continue

            if result is None:
                transaction_logger.warning(f"Skipping file due to parsing issues: {filename}")
                bad_files += 1
                continue

            json_transcript, participants, symbol, companyname, date, quarter, year = result

            if not all([json_transcript, participants, symbol, companyname, date, quarter, year]):
                transaction_logger.warning(f"Skipping file due to missing data: {filename}, Symbol: {symbol}, Company: {companyname}, Date: {date}")
                bad_files += 1
                continue

            existing_call = Call.query.filter_by(
                symbol=symbol,
                companyname=companyname,
                date=date,
                quarter=quarter,
                year=year,
            ).first()

            if existing_call:
                #transaction_logger.info(f"Skipping existing call: {filename}, Symbol: {symbol}, Company: {companyname}, Date: {date}")
                num_skipped += 1
                continue

            new_call = Call(
                symbol=symbol,
                json_transcript=json_transcript,
                summary="",
                companyname=companyname,
                date=date,
                quarter=quarter,
                year=year,
            )

            for participant in participants:
                new_participant = Participant(
                    name=participant["name"],
                    role=participant["role"],
                )
                new_call.participants.append(new_participant)

            db.session.add(new_call)
            transaction_logger.info(f"Processed Sucessfully: {filename}, Symbol: {symbol}, Company: {companyname}, Date: {date}")
              
            if symbol not in processed_symbols:
                processed_symbols.append(symbol)

    db.session.commit()
    flash(
        f"Processed {num_processed - 1} files. Skipped {num_skipped} existing files, found {bad_files} bad files.",
        "success")
    return processed_symbols


def parse_transcript(file_path):
    with open(file_path, "r", encoding="utf-8") as file:
        soup = BeautifulSoup(file, "html.parser")

    # Check if the page is unavailable
    if soup.title and "Page Unavailable" in soup.title.text:
        return None, None, None, None, None, None, None

    transcript_elements = soup.find_all("div", class_="speech")
    transcript = []  # Initialize transcript as a list to store speaker and content

    for element in transcript_elements:
        speaker = element.find('div', class_='participant-name').text
        content = element.find('div', class_='content').text.strip()
        transcript.append({"speaker": speaker, "content": content})  # Add speaker and content to the list
 # Convert the transcript list to JSON
    transcript_json = json.dumps(transcript)

    participant_table = soup.find("table", class_="participant-list")
    participants = []
    if participant_table:
        for row in participant_table.find_all("tr"):
            name = row.find("td", class_="name").text.strip()
            role = row.find("td", class_="role").text.strip()
            participants.append({"name": name, "role": role})

     # Extract the symbol from the canonical URL
    symbol = None
    canonical_link = soup.find("link", {"rel": "canonical"})
    if canonical_link:
        canonical_url = canonical_link.get("href")
        if canonical_url:
            symbol = canonical_url.rstrip('/').split('/')[-1]  # Extract the second-to-last part

        title_text = soup.title.text
        companyname = title_text.split("|")[1].strip()  # Get the second element and strip whitespace

        banner_crumbs = soup.find_all("span", class_="banner-crumb")
        if len(banner_crumbs) >= 2:
            date = banner_crumbs[0].text.strip()
            date = convert_date_to_iso(date)
            period_full = banner_crumbs[1].text.strip()
            year = period_full.split()[0]
            quarter =(period_full.split()[1])
        else:
            date = None
            period_full = None
            year = None
            quarter = None
            companyname = None
            symbol = None
            transcript_json = None # Reset the transcript_json to None if any of the required fields are missing
            participants = None # Reset the participants to None if any of the required fields are missing
        return transcript_json, participants, symbol, companyname, date, quarter, year

    

def check_favorites(db, processed_symbols):
    """Checks if any processed symbols match user favorites in the database.

    Args:
        db: The SQLAlchemy database session.
        processed_symbols: A list of symbols that were processed.

    Returns:
        A list of tuples, where each tuple contains:
            - The user's ID
            - The user's username
            - The matching symbol
            - The ID of the Call object
    """
    matching_favorites = []
    for user in db.session.query(User).all():
        if user.favorites:
            for symbol in processed_symbols:
                if symbol in user.favorites:
                    # Find the Call object for this symbol
                    call = db.session.query(Call).filter_by(symbol=symbol).first() 
                    if call:
                        matching_favorites.append((user.id, user.username, symbol, call.id))
    return matching_favorites